﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Data;
using System.Linq;
using System.Text;
using System.Windows.Forms;

using Advent.aiDevices;

namespace Advent.Controls
{
    public partial class FrequencyCtrl : UserControl
    {
        Frequency F = Frequency.InHz(1000);

        /// <summary>
        /// This event is raised whenver the frequency is changed due to user actions.
        /// </summary>
        public event EventHandler FrequencyChanged;

        public FrequencyCtrl()
        {
            InitializeComponent();
            txt_Freq.Validating += new CancelEventHandler(txt_Freq_Validating);
            txt_Freq.KeyUp += new KeyEventHandler(txt_Freq_KeyUp);
            this.Resize += new EventHandler(FrequencyCtrl_Resize);
        }

        void FrequencyCtrl_Resize(object sender, EventArgs e)
        {
            txt_Freq.Location = new Point(0, 0);
            lbl_Hz.Left = Width - lbl_Hz.Width;
            lbl_Hz.Top = (Height - lbl_Hz.Height) / 2;
            txt_Freq.Size = new Size(Width - lbl_Hz.Width - 2, txt_Freq.Height);
        }

        private void FrequencyCtrl_Load(object sender, EventArgs e)
        {
            FrequencyCtrl_Resize(this, EventArgs.Empty);
        }

        /// <summary>
        /// Called when a key is up within the text box
        /// </summary>
        void txt_Freq_KeyUp(object sender, KeyEventArgs e)
        {
            if (e.KeyCode != Keys.Return) return;
            try
            {
                UpdateFrequency();
            }
            catch (Exception ex)
            {
                MessageBox.Show(ex.Message);
            }
            finally
            {
                RefreshControls();
            }
        }

        /// <summary>
        /// Called when the textbox is validating
        /// </summary>
        void txt_Freq_Validating(object sender, CancelEventArgs e)
        {
            try
            {
                UpdateFrequency();

            }
            catch (Exception ex)
            {
                MessageBox.Show(ex.Message);
                e.Cancel = true;
            }
            finally
            {
                RefreshControls();
            }
        }

        /// <summary>
        /// This updates the internal frequency with the parsed contents
        /// and detects changes
        /// </summary>
        private void UpdateFrequency()
        {
            Frequency Freq = Parse();
            if (F != Freq)
            {
                F = Freq;
                OnChanged();
            }
        }
        /// <summary>
        /// This method simply raises the FrequencyChanged event
        /// </summary>
        private void OnChanged()
        {
            if (FrequencyChanged == null) return;
            FrequencyChanged(this, EventArgs.Empty);
        }


        /// <summary>
        /// This parses the current contents of the controls and returns the corresponding Frequency 
        /// </summary>
        /// <returns></returns>
        private Frequency Parse()
        {
            double Value;
            if (!double.TryParse(txt_Freq.Text, out Value)) throw new Exception("Invalid frequency");
            return Frequency.InHz(Value);
        }

        /// <summary>
        /// This sets/gets the frequency displayed within this control
        /// </summary>
        public Frequency Freq
        {
            get
            {
                return F;
            }
            set
            {
                if (value == null) throw new NullReferenceException();
                F = value;
                RefreshControls();
            }
        }

        /// <summary>
        /// This updates the display with the internal frequency value
        /// </summary>
        private void RefreshControls()
        {
            try
            {
                txt_Freq.Text = Advent.aiLibrary.UnitsFormatter.FormatSigFigs(F.Value, 1, 1, true); 
            }
            catch { }
        }


    }
}
